/**
 * Основной JavaScript файл для шаблона sport38-kant
 * В стиле kant.ru - современный, чистый код
 */

(function() {
    'use strict';
    
    // Инициализация при загрузке DOM
    document.addEventListener('DOMContentLoaded', function() {
        console.log('Sport38 Kant template loaded');
        
        initModals();
        initSearch();
        initMobileMenu();
        initHeroSlider();
        initTabs();
        initAccordion();
        initProductGallery();
        initCart();
        initCompare();
        initDropdown();
    });
    
    /**
     * Инициализация модальных окон
     */
    function initModals() {
        const modalTriggers = document.querySelectorAll('[data-modal]');
        const modals = document.querySelectorAll('.modal');
        const modalCloses = document.querySelectorAll('.modal-close, .modal-backdrop');
        
        // Открытие модалки
        modalTriggers.forEach(trigger => {
            trigger.addEventListener('click', function(e) {
                e.preventDefault();
                const modalId = this.getAttribute('data-modal');
                const modal = document.getElementById(modalId);
                if (modal) {
                    modal.classList.add('active');
                    document.body.style.overflow = 'hidden';
                }
            });
        });
        
        // Закрытие модалки
        modalCloses.forEach(close => {
            close.addEventListener('click', function() {
                const modal = this.closest('.modal') || document.querySelector('.modal.active');
                if (modal) {
                    modal.classList.remove('active');
                    document.body.style.overflow = '';
                }
            });
        });
        
        // Закрытие по ESC
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                const activeModal = document.querySelector('.modal.active');
                if (activeModal) {
                    activeModal.classList.remove('active');
                    document.body.style.overflow = '';
                }
            }
        });
    }
    
    /**
     * Инициализация поиска
     */
    function initSearch() {
        const searchForm = document.querySelector('.search-form');
        if (searchForm) {
            searchForm.addEventListener('submit', function(e) {
                const input = this.querySelector('.search-input');
                if (!input || !input.value.trim()) {
                    e.preventDefault();
                    input.focus();
                }
            });
        }
    }
    
    /**
     * Инициализация мобильного меню
     */
    function initMobileMenu() {
        const menuToggle = document.querySelector('[data-menu-toggle]');
        const mobileMenu = document.querySelector('[data-mobile-menu]');
        const menuOverlay = document.querySelector('[data-menu-overlay]');
        
        if (menuToggle && mobileMenu) {
            menuToggle.addEventListener('click', function() {
                const isActive = mobileMenu.classList.contains('active');
                
                if (isActive) {
                    closeMobileMenu();
                } else {
                    openMobileMenu();
                }
            });
            
            // Закрытие по клику на overlay
            if (menuOverlay) {
                menuOverlay.addEventListener('click', closeMobileMenu);
            }
            
            // Закрытие по ESC
            document.addEventListener('keydown', function(e) {
                if (e.key === 'Escape' && mobileMenu.classList.contains('active')) {
                    closeMobileMenu();
                }
            });
        }
        
        function openMobileMenu() {
            if (mobileMenu) mobileMenu.classList.add('active');
            if (menuToggle) menuToggle.classList.add('active');
            if (menuOverlay) menuOverlay.classList.add('active');
            document.body.style.overflow = 'hidden';
        }
        
        function closeMobileMenu() {
            if (mobileMenu) mobileMenu.classList.remove('active');
            if (menuToggle) menuToggle.classList.remove('active');
            if (menuOverlay) menuOverlay.classList.remove('active');
            document.body.style.overflow = '';
        }
    }
    
    /**
     * Инициализация Hero слайдера
     */
    function initHeroSlider() {
        const slider = document.querySelector('.hero-slider');
        if (!slider) return;
        
        const slides = slider.querySelectorAll('.hero-slide');
        const indicators = slider.querySelectorAll('.hero-indicator');
        const prevBtn = slider.querySelector('.hero-nav.prev');
        const nextBtn = slider.querySelector('.hero-nav.next');
        
        if (slides.length <= 1) return;
        
        let currentSlide = 0;
        let autoplayInterval;
        
        function showSlide(index) {
            slides.forEach((slide, i) => {
                slide.style.display = i === index ? 'flex' : 'none';
            });
            
            indicators.forEach((indicator, i) => {
                indicator.classList.toggle('active', i === index);
            });
            
            currentSlide = index;
        }
        
        function nextSlide() {
            const next = (currentSlide + 1) % slides.length;
            showSlide(next);
        }
        
        function prevSlide() {
            const prev = (currentSlide - 1 + slides.length) % slides.length;
            showSlide(prev);
        }
        
        // Навигация по индикаторам
        indicators.forEach((indicator, index) => {
            indicator.addEventListener('click', () => showSlide(index));
        });
        
        // Кнопки навигации
        if (nextBtn) nextBtn.addEventListener('click', nextSlide);
        if (prevBtn) prevBtn.addEventListener('click', prevSlide);
        
        // Автоплей
        function startAutoplay() {
            autoplayInterval = setInterval(nextSlide, 5000);
        }
        
        function stopAutoplay() {
            if (autoplayInterval) {
                clearInterval(autoplayInterval);
            }
        }
        
        slider.addEventListener('mouseenter', stopAutoplay);
        slider.addEventListener('mouseleave', startAutoplay);
        
        // Инициализация
        showSlide(0);
        startAutoplay();
    }
    
    /**
     * Обновление счетчика корзины
     */
    function updateCartCount(count) {
        const cartCount = document.querySelector('.cart-count');
        if (cartCount) {
            cartCount.textContent = count || 0;
            if (count > 0) {
                cartCount.style.display = 'flex';
            } else {
                cartCount.style.display = 'none';
            }
        }
    }
    
    /**
     * Добавление товара в корзину
     */
    function addToCart(productId, quantity) {
        // Здесь будет интеграция с Bitrix корзиной
        console.log('Add to cart:', productId, quantity);
        // TODO: AJAX запрос к Bitrix API
    }
    
    /**
     * Инициализация табов
     */
    function initTabs() {
        const tabsContainers = document.querySelectorAll('.tabs');
        
        tabsContainers.forEach(container => {
            const tabs = container.querySelectorAll('.tabs__item');
            const contents = container.parentElement.querySelectorAll('[data-tab-content]');
            
            tabs.forEach(tab => {
                tab.addEventListener('click', function() {
                    const targetTab = this.getAttribute('data-tab');
                    
                    // Убираем активный класс со всех табов
                    tabs.forEach(t => t.classList.remove('active'));
                    contents.forEach(c => c.classList.remove('active'));
                    
                    // Добавляем активный класс к выбранному табу и контенту
                    this.classList.add('active');
                    const targetContent = container.parentElement.querySelector(`[data-tab-content="${targetTab}"]`);
                    if (targetContent) {
                        targetContent.classList.add('active');
                    }
                });
            });
        });
    }
    
    /**
     * Инициализация аккордеона
     */
    function initAccordion() {
        const accordionHeaders = document.querySelectorAll('.accordion-header');
        
        accordionHeaders.forEach(header => {
            header.addEventListener('click', function() {
                const content = this.nextElementSibling;
                const isActive = this.classList.contains('active');
                
                // Закрываем все аккордеоны в этой группе
                const accordion = this.closest('.accordion');
                accordion.querySelectorAll('.accordion-header').forEach(h => {
                    h.classList.remove('active');
                    h.nextElementSibling.classList.remove('active');
                });
                
                // Открываем текущий, если он был закрыт
                if (!isActive) {
                    this.classList.add('active');
                    content.classList.add('active');
                }
            });
        });
    }
    
    /**
     * Инициализация галереи товара
     */
    function initProductGallery() {
        const gallery = document.querySelector('.product-gallery');
        if (!gallery) return;
        
        const mainImage = gallery.querySelector('#gallery-main');
        const thumbs = gallery.querySelectorAll('.product-gallery__thumb');
        
        thumbs.forEach(thumb => {
            thumb.addEventListener('click', function() {
                const imageSrc = this.getAttribute('data-image');
                if (mainImage && imageSrc) {
                    mainImage.src = imageSrc;
                }
                
                // Обновляем активную миниатюру
                thumbs.forEach(t => t.classList.remove('active'));
                this.classList.add('active');
            });
        });
    }
    
    /**
     * Инициализация корзины
     */
    function initCart() {
        // Изменение количества
        const quantityControls = document.querySelectorAll('.quantity-btn, .quantity-control button');
        quantityControls.forEach(btn => {
            btn.addEventListener('click', function() {
                const input = this.closest('.quantity-control')?.querySelector('input') || 
                            this.closest('.cart-item__quantity')?.querySelector('.quantity-input');
                if (!input) return;
                
                const currentValue = parseInt(input.value) || 1;
                if (this.classList.contains('quantity-decrease') || this.textContent === '-') {
                    if (currentValue > 1) {
                        input.value = currentValue - 1;
                    }
                } else {
                    input.value = currentValue + 1;
                }
                
                // TODO: Обновление через AJAX
            });
        });
        
        // Удаление товара
        const removeBtns = document.querySelectorAll('[data-remove]');
        removeBtns.forEach(btn => {
            btn.addEventListener('click', function() {
                const itemId = this.getAttribute('data-remove');
                if (confirm('Удалить товар из корзины?')) {
                    // TODO: AJAX удаление
                    this.closest('.cart-item')?.remove();
                }
            });
        });
        
        // Добавление в корзину
        const addToCartBtns = document.querySelectorAll('[data-add-to-cart]');
        addToCartBtns.forEach(btn => {
            btn.addEventListener('click', function() {
                const productId = this.getAttribute('data-add-to-cart');
                const quantity = document.querySelector('input[name="quantity"]')?.value || 1;
                addToCart(productId, quantity);
            });
        });
    }
    
    /**
     * Инициализация страницы сравнения
     */
    function initCompare() {
        // Удаление товара из сравнения
        const removeBtns = document.querySelectorAll('[data-remove]');
        removeBtns.forEach(btn => {
            btn.addEventListener('click', function() {
                const itemId = this.getAttribute('data-remove');
                if (confirm('Удалить товар из сравнения?')) {
                    // TODO: AJAX удаление
                    const productCol = this.closest('th, td')?.parentElement;
                    if (productCol) {
                        productCol.remove();
                    }
                }
            });
        });
        
        // Очистка сравнения
        const clearBtn = document.querySelector('[data-clear-compare]');
        if (clearBtn) {
            clearBtn.addEventListener('click', function() {
                if (confirm('Очистить все сравнение?')) {
                    // TODO: AJAX очистка
                    window.location.reload();
                }
            });
        }
        
        // Добавление всех в корзину
        const addAllBtn = document.querySelector('[data-compare-all-to-cart]');
        if (addAllBtn) {
            addAllBtn.addEventListener('click', function() {
                const productIds = Array.from(document.querySelectorAll('[data-add-to-cart]'))
                    .map(btn => btn.getAttribute('data-add-to-cart'));
                // TODO: AJAX добавление всех товаров
                console.log('Add all to cart:', productIds);
            });
        }
        
        // Печать сравнения
        const printBtn = document.querySelector('[data-print-compare]');
        if (printBtn) {
            printBtn.addEventListener('click', function() {
                window.print();
            });
        }
    }
    
    /**
     * Инициализация dropdown меню
     */
    function initDropdown() {
        const dropdowns = document.querySelectorAll('.dropdown');
        
        dropdowns.forEach(dropdown => {
            const toggle = dropdown.querySelector('.dropdown__toggle');
            const menu = dropdown.querySelector('.dropdown__menu');
            
            if (toggle && menu) {
                toggle.addEventListener('click', function(e) {
                    e.stopPropagation();
                    const isActive = dropdown.classList.contains('active');
                    
                    // Закрываем все другие dropdown
                    dropdowns.forEach(d => {
                        if (d !== dropdown) {
                            d.classList.remove('active');
                        }
                    });
                    
                    // Переключаем текущий
                    dropdown.classList.toggle('active', !isActive);
                });
            }
        });
        
        // Закрытие по клику вне dropdown
        document.addEventListener('click', function() {
            dropdowns.forEach(d => d.classList.remove('active'));
        });
    }
    
    // Экспорт функций для использования в других скриптах
    window.Sport38Kant = {
        updateCartCount: updateCartCount,
        addToCart: addToCart
    };
})();

