<?php
/**
 * Пользовательский тип свойства "Список с возможностью добавления"
 * Оптимизирован для PHP 8.1+
 * 
 * @package Sport38\Kant
 */

class CSport38KantIBlockPropertyList
{
    private static bool $scriptsIncluded = false;
    private static array $enumCache = [];

    /**
     * Получить описание типа свойства
     * 
     * @return array
     */
    public static function GetUserTypeDescription(): array
    {
        return [
            'PROPERTY_TYPE' => 'L',
            'USER_TYPE' => "sport38_list",
            'DESCRIPTION' => "Список с возможностью добавления",
            'GetPropertyFieldHtml' => [__CLASS__, 'GetPropertyFieldHtml'],
        ];
    }

    /**
     * Получить HTML поля свойства
     * 
     * @param array $arProperty Свойство
     * @param array $value Значение
     * @param array $strHTMLControlName Имена контролов
     * @return string HTML
     */
    public static function GetPropertyFieldHtml(array $arProperty, array $value, array $strHTMLControlName): string
    {
        // Нормализация значений
        $values = is_array($value) && !isset($value["VALUE"]) ? $value : [$value];
        foreach ($values as $key => $val) {
            if (is_array($val) && isset($val["VALUE"])) {
                $values[$key] = $val["VALUE"];
            }
        }

        $id = (int)($arProperty["ID"] ?? 0);
        $name = $strHTMLControlName['VALUE'] ?? '';
        $htmlID = "list_prop" . randString();
        $bNoValue = true;

        // Получение списка значений
        $propEnums = self::getEnum($id);
        $options = '';

        foreach ($propEnums as $arEnum) {
            $sel = in_array($arEnum["ID"], $values, true);
            if ($sel) {
                $bNoValue = false;
            }
            $options .= '<option value="' . htmlspecialcharsbx($arEnum["ID"]) . '"' . ($sel ? ' selected' : '') . '>'
                . htmlspecialcharsex($arEnum["VALUE"]) . '</option>';
        }

        $res = '<select name="' . htmlspecialcharsbx($name) . '" size="' . (int)($arProperty["ROW_COUNT"] ?? 5) . '" id="' . $htmlID . '">'
            . '<option value=""' . ($bNoValue ? ' selected' : '') . '>' . htmlspecialcharsex(GetMessage("IBLOCK_AT_PROP_NA")) . '</option>'
            . $options
            . '</select>';

        $res .= '<input type="button" value="+" onclick="addNewListValue(\'' . $htmlID . '\', ' . $id . ', ' . (int)($arProperty["IBLOCK_ID"] ?? 0) . ')">';
        $res .= self::includeJS();

        return $res;
    }

    /**
     * Получить список значений свойства
     * 
     * @param int $id ID свойства
     * @return array Массив значений
     */
    private static function getEnum(int $id): array
    {
        if (!isset(self::$enumCache[$id]) || !is_array(self::$enumCache[$id])) {
            self::$enumCache[$id] = [];

            $db = \CIBlockProperty::GetPropertyEnum($id);
            while ($item = $db->Fetch()) {
                self::$enumCache[$id][] = $item;
            }
        }

        return self::$enumCache[$id];
    }

    /**
     * Подключить JavaScript
     * 
     * @return string HTML скрипта
     */
    private static function includeJS(): string
    {
        if (self::$scriptsIncluded) {
            return '';
        }

        self::$scriptsIncluded = true;
        \CJSCore::Init(['jquery']);

        ob_start();
        ?>
        <script>
            function addNewListValue(htmlID, propID, iblockID) {
                var newName = prompt("Введите новое значение");
                
                if (newName) {
                    $.ajax({
                        url: '/bitrix/tools/sport38_list_add.php',
                        type: 'post',
                        data: {
                            prop_id: propID,
                            iblock_id: iblockID,
                            value: newName,
                            sessid: BX.bitrix_sessid()
                        },
                        success: function(response) {
                            if (response && response.ID) {
                                var select = document.getElementById(htmlID);
                                var option = document.createElement('option');
                                option.value = response.ID;
                                option.text = newName;
                                option.selected = true;
                                select.appendChild(option);
                            }
                        }
                    });
                }
            }
        </script>
        <?php
        return ob_get_clean();
    }
}

