<?php
/**
 * Класс для работы с бонусами
 * Оптимизирован для PHP 8.1+ с улучшенным кешированием
 * 
 * @package Sport38\Kant
 */

namespace Sport38\Kant;

class Bonus
{
    private static bool $fetched = false;
    private static array $cache = [];
    private static ?int $iblockId = null;
    private const CACHE_TIME = 3600;
    private const CACHE_DIR = "sport38.kant.bonus";

    /**
     * Рассчитать бонусы
     * 
     * @param int $bonusID ID бонусной карты
     * @param float $price Цена
     * @param float $priceFull Полная цена (для проверки)
     * @return int Количество бонусов
     */
    public static function calc(int $bonusID, float $price, float $priceFull = 0): int
    {
        if ($price <= 0 || $price < $priceFull || $bonusID <= 0) {
            return 0;
        }

        $bonus = self::fetch($bonusID);
        if (!$bonus || !is_array($bonus)) {
            return 0;
        }

        $fold = max(1, (int)($bonus['FOLD'] ?? 1));
        $foldPrice = $price - ($price % $fold);
        $percent = (float)($bonus['PERCENT'] ?? 0);

        return (int)floor($foldPrice * $percent);
    }

    /**
     * Получить данные бонусной карты
     * 
     * @param int|array|false $bonusID ID бонусной карты или массив ID, или false для всех
     * @return array|false Данные бонусной карты или массив карт
     */
    public static function fetch($bonusID = false)
    {
        if (!self::$fetched) {
            self::loadFromCache();
        }

        if ($bonusID === false) {
            return self::$cache;
        }

        if (is_array($bonusID)) {
            $result = [];
            foreach ($bonusID as $id) {
                $id = (int)$id;
                if ($id > 0 && isset(self::$cache[$id])) {
                    $result[$id] = self::$cache[$id];
                }
            }
            return $result;
        }

        $bonusID = (int)$bonusID;
        return $bonusID > 0 && isset(self::$cache[$bonusID]) ? self::$cache[$bonusID] : false;
    }

    /**
     * Загрузить данные из кеша или базы
     */
    private static function loadFromCache(): void
    {
        $cache = new \CPHPCache();
        $cacheId = "bonus_cards_" . self::getIBlockID();

        if ($cache->InitCache(self::CACHE_TIME, $cacheId, self::CACHE_DIR)) {
            self::$cache = $cache->GetVars();
            self::$fetched = true;
            return;
        }

        if (!\Bitrix\Main\Loader::includeModule("iblock")) {
            self::$fetched = true;
            return;
        }

        $elements = [];
        $iblockId = self::getIBlockID();

        if ($iblockId > 0) {
            $select = [
                "ID",
                "ACTIVE",
                "ACTIVE_FROM",
                "ACTIVE_TO",
                "NAME",
                "PROPERTY_PERCENT",
                "PROPERTY_FOLD",
            ];

            $filter = [
                "IBLOCK_ID" => $iblockId,
                "ACTIVE" => "Y"
            ];

            $dbElement = \CIBlockElement::GetList(
                ["SORT" => "ASC", "ID" => "ASC"],
                $filter,
                false,
                false,
                $select
            );

            while ($element = $dbElement->Fetch()) {
                // Проверка активности по датам
                if (!self::isActiveByDate($element)) {
                    continue;
                }

                $elements[(int)$element["ID"]] = [
                    "ID" => (int)$element["ID"],
                    "ACTIVE_FROM" => $element["ACTIVE_FROM"] ?? null,
                    "ACTIVE_TO" => $element["ACTIVE_TO"] ?? null,
                    "NAME" => $element["NAME"] ?? "",
                    "PERCENT" => (floatval($element["PROPERTY_PERCENT_VALUE"] ?? 0)) / 100,
                    "FOLD" => (int)($element["PROPERTY_FOLD_VALUE"] ?? 1),
                ];
            }
        }

        $cache->StartDataCache();
        $cache->EndDataCache($elements);

        if (defined("BX_COMP_MANAGED_CACHE")) {
            global $CACHE_MANAGER;
            $CACHE_MANAGER->StartTagCache(self::CACHE_DIR);
            $CACHE_MANAGER->RegisterTag("iblock_id_" . $iblockId);
            $CACHE_MANAGER->EndTagCache();
        }

        self::$cache = $elements;
        self::$fetched = true;
    }

    /**
     * Проверить активность по датам
     * 
     * @param array $element Элемент инфоблока
     * @return bool
     */
    private static function isActiveByDate(array $element): bool
    {
        $now = time();
        
        if (!empty($element["ACTIVE_FROM"])) {
            $from = MakeTimeStamp($element["ACTIVE_FROM"]);
            if ($from > $now) {
                return false;
            }
        }
        
        if (!empty($element["ACTIVE_TO"])) {
            $to = MakeTimeStamp($element["ACTIVE_TO"]);
            if ($to < $now) {
                return false;
            }
        }
        
        return true;
    }

    /**
     * Получить ID инфоблока бонусных карт
     * 
     * @return int ID инфоблока
     */
    public static function getIBlockID(): int
    {
        if (self::$iblockId === null) {
            self::$iblockId = defined("BONUS_CARD_IBLOCK_ID") ? (int)BONUS_CARD_IBLOCK_ID : 0;
        }
        return self::$iblockId;
    }

    /**
     * Очистить кеш
     */
    public static function clearCache(): void
    {
        self::$fetched = false;
        self::$cache = [];
        
        $cache = new \CPHPCache();
        $cache->CleanDir(self::CACHE_DIR);
        
        if (defined("BX_COMP_MANAGED_CACHE")) {
            global $CACHE_MANAGER;
            $CACHE_MANAGER->ClearByTag("iblock_id_" . self::getIBlockID());
        }
    }
}

