<?php
/**
 * Класс для работы с корзиной
 * Оптимизирован для PHP 8.1+ с улучшенными запросами
 * 
 * @package Sport38\Kant
 */

namespace Sport38\Kant;

class Cart
{
    private static ?array $cacheReady = null;
    private const CACHE_TIME = 300; // 5 минут

    /**
     * Получить список ID товаров в корзине
     * 
     * @param bool $useCache Использовать кеш
     * @return array Массив ID товаров
     */
    public static function getReady(bool $useCache = true): array
    {
        if ($useCache && self::$cacheReady !== null) {
            return self::$cacheReady;
        }

        $ids = [];

        if (!\Bitrix\Main\Loader::includeModule("sale")) {
            return $ids;
        }

        try {
            $fuserId = \CSaleBasket::GetBasketUserID(true);
            
            $db = \CSaleBasket::GetList(
                [],
                [
                    "FUSER_ID" => $fuserId,
                    "LID" => SITE_ID,
                    "ORDER_ID" => "NULL",
                    "CAN_BUY" => "Y",
                    "DELAY" => "N",
                    "SUBSCRIBE" => "N"
                ],
                false,
                false,
                ["PRODUCT_ID"]
            );

            while ($item = $db->Fetch()) {
                $productId = (int)($item["PRODUCT_ID"] ?? 0);
                if ($productId > 0 && !in_array($productId, $ids, true)) {
                    $ids[] = $productId;
                }
            }

            // Получить связанные товары (CML2_LINK)
            if (!empty($ids) && \Bitrix\Main\Loader::includeModule("iblock")) {
                $ids = self::resolveLinkedProducts($ids);
            }

            self::$cacheReady = $ids;
        } catch (\Exception $e) {
            // Логирование ошибки (если нужно)
            return [];
        }

        return $ids;
    }

    /**
     * Разрешить связанные товары (CML2_LINK)
     * 
     * @param array $ids Массив ID товаров
     * @return array Массив ID с разрешенными связями
     */
    private static function resolveLinkedProducts(array $ids): array
    {
        if (empty($ids)) {
            return [];
        }

        $filter = [
            "ID" => $ids,
            "!PROPERTY_CML2_LINK" => false
        ];

        $dbLink = \CIBlockElement::GetList(
            [],
            $filter,
            false,
            false,
            [
                "IBLOCK_ID",
                "ID",
                "PROPERTY_CML2_LINK"
            ]
        );

        $linkedIds = [];
        $toRemove = [];

        while ($item = $dbLink->Fetch()) {
            $linkId = (int)$item["ID"];
            $linkValue = $item["PROPERTY_CML2_LINK_VALUE"] ?? null;

            if (empty($linkValue)) {
                continue;
            }

            // Обработка множественных значений
            $linkValues = is_array($linkValue) ? $linkValue : [$linkValue];

            foreach ($linkValues as $linkedId) {
                $linkedId = (int)$linkedId;
                if ($linkedId > 0) {
                    $linkedIds[] = $linkedId;
                    $toRemove[] = $linkId; // Удалить оффер, оставить основной товар
                }
            }
        }

        // Объединить и удалить дубликаты
        $result = array_merge($ids, $linkedIds);
        $result = array_diff($result, $toRemove);
        $result = array_unique($result);
        $result = array_values($result); // Переиндексация

        return $result;
    }

    /**
     * Очистить кеш
     */
    public static function clearCache(): void
    {
        self::$cacheReady = null;
    }

    /**
     * Проверить, есть ли товар в корзине
     * 
     * @param int $productId ID товара
     * @return bool
     */
    public static function isInCart(int $productId): bool
    {
        $ids = self::getReady();
        return in_array($productId, $ids, true);
    }

    /**
     * Получить количество товаров в корзине
     * 
     * @return int Количество товаров
     */
    public static function getCount(): int
    {
        $ids = self::getReady();
        return count($ids);
    }
}

