<?php
/**
 * Класс для работы с флагами скидок
 * Оптимизирован для PHP 8.1+
 * 
 * @package Sport38\Kant
 */

namespace Sport38\Kant;

class DiscountFlag
{
    private static ?int $lastQueryId = null;
    private static array $discountFlags = [];
    private static array $productDiscounts = [];

    /**
     * Кешировать ID товара для последующего сохранения результата
     * 
     * @param int $productID ID товара
     * @return bool
     */
    public static function cacheQuery(int $productID): bool
    {
        self::$lastQueryId = $productID;
        return true;
    }

    /**
     * Сохранить результат расчета скидок
     * 
     * @param array $result Массив скидок
     * @return bool
     */
    public static function saveResult(array $result): bool
    {
        if (empty($result) || self::$lastQueryId === null) {
            self::$lastQueryId = null;
            return false;
        }

        $productId = self::$lastQueryId;

        if (!isset(self::$productDiscounts[$productId])) {
            self::$productDiscounts[$productId] = [];
        }

        foreach ($result as $discount) {
            $discountId = (int)($discount["ID"] ?? 0);
            if ($discountId > 0) {
                self::$productDiscounts[$productId][] = $discountId;
            }
        }

        self::$lastQueryId = null;
        return true;
    }

    /**
     * Получить флаги скидок для товара
     * 
     * @param int $productID ID товара
     * @return array|false Массив флагов или false
     */
    public static function getFlagsByProduct(int $productID)
    {
        if ($productID <= 0 || !isset(self::$productDiscounts[$productID])) {
            return false;
        }

        $result = [];
        $discountIds = self::$productDiscounts[$productID];

        foreach ($discountIds as $discountId) {
            $flags = self::getFlags($discountId);
            if ($flags && is_array($flags)) {
                $result = array_merge($result, $flags);
            }
        }

        return !empty($result) ? array_unique($result) : false;
    }

    /**
     * Получить флаги для скидки
     * 
     * @param int $discountID ID скидки
     * @return array|false Массив флагов или false
     */
    public static function getFlags(int $discountID)
    {
        if ($discountID <= 0) {
            return false;
        }

        // Проверка кеша
        if (isset(self::$discountFlags[$discountID])) {
            return self::$discountFlags[$discountID];
        }

        if (!\Bitrix\Main\Loader::includeModule('catalog')) {
            return false;
        }

        $result = false;

        $query = \CCatalogDiscount::GetList(
            [],
            ["ID" => $discountID],
            false,
            ["nTopCount" => 1],
            ["NOTES"]
        );

        if ($discount = $query->Fetch()) {
            $text = $discount["NOTES"] ?? "";
            
            if (!empty($text) && preg_match_all('/FLAG_([A-Z0-9]+)/', $text, $matches)) {
                $flags = $matches[1] ?? [];
                $result = is_array($flags) ? $flags : [$flags];
            }
        }

        // Сохранение в кеш
        self::$discountFlags[$discountID] = $result;

        return $result;
    }

    /**
     * Очистить кеш
     */
    public static function clearCache(): void
    {
        self::$lastQueryId = null;
        self::$discountFlags = [];
        self::$productDiscounts = [];
    }
}

