<?php
/**
 * Класс для работы с почтой
 * Оптимизирован для PHP 8.1+
 * 
 * @package Sport38\Kant
 */

namespace Sport38\Kant;

class Mail
{
    private static string $cachePath = "/email/";
    private static string $cacheID = "my";
    private static int $cacheTTL = 360000;

    /**
     * Обработать отправку письма
     * 
     * @param array $fields Поля письма
     * @param array $dbResult Результат
     * @return void
     */
    public static function sent(array &$fields, array &$dbResult): void
    {
        $html = self::getTemplate();

        if (
            strpos($dbResult["MESSAGE"] ?? "", "<!") !== 0 &&
            strpos($html, "#CONTENT#") !== false
        ) {
            if (($dbResult["BODY_TYPE"] ?? "") !== "html") {
                $dbResult["MESSAGE"] = nl2br($dbResult["MESSAGE"] ?? "");
            }

            $dbResult["MESSAGE"] = str_replace("#CONTENT#", $dbResult["MESSAGE"] ?? "", $html);
            $dbResult["BODY_TYPE"] = "html";
        }
    }

    /**
     * Получить шаблон письма
     * 
     * @param bool $ignoreCache Игнорировать кеш
     * @param bool $byAgent Вызывается агентом
     * @return string HTML шаблона
     */
    public static function getTemplate(bool $ignoreCache = false, bool $byAgent = false): string
    {
        $cache = new \CPHPCache();

        if ($cache->InitCache(self::$cacheTTL, self::$cacheID, self::$cachePath) && !$ignoreCache) {
            return $cache->GetVars() ?: "";
        }

        $domain = \COption::GetOptionString("main", "server_name", "");
        if (empty($domain)) {
            return "";
        }

        $subscribeKey = defined("SUBSCRIBE_KEY") ? SUBSCRIBE_KEY : "";
        $url = "https://" . $domain . "/email/template/?" . http_build_query(["SUBSCRIBE_KEY" => $subscribeKey]);

        $html = @file_get_contents($url);
        if (!$html) {
            return "";
        }

        $html = self::absolutizeUrl($html, $domain);

        $cache->StartDataCache();
        $cache->EndDataCache($html);

        if (!$byAgent) {
            $agentName = '\Sport38\Kant\Mail::agentUpdateTemplate();';
            \CAgent::RemoveAgent($agentName, "sport38.kant");
            \CAgent::AddAgent($agentName, "sport38.kant", "Y", 3600);
        }

        return $html;
    }

    /**
     * Агент обновления шаблона
     * 
     * @return string
     */
    public static function agentUpdateTemplate(): string
    {
        self::getTemplate(true, true);
        return '\Sport38\Kant\Mail::agentUpdateTemplate();';
    }

    /**
     * Обработать данные для отписки
     * 
     * @param array $fields Поля письма
     * @return array
     */
    public static function unsubscribeData(array $fields): array
    {
        if (!\Bitrix\Main\Loader::includeModule("subscribe")) {
            return $fields;
        }

        $email = $fields["EMAIL"] ?? "";
        if (empty($email)) {
            return $fields;
        }

        $rsSubscriber = \CSubscription::GetByEmail($email);
        if ($arSubscriber = $rsSubscriber->Fetch()) {
            $body = $fields["BODY"] ?? "";
            $body = str_replace("#CONFIRM_CODE#", $arSubscriber["CONFIRM_CODE"] ?? "", $body);
            $body = str_replace("#ID#", (string)($arSubscriber["ID"] ?? ""), $body);
            $fields["BODY"] = $body;
        }

        return $fields;
    }

    /**
     * Абсолютизировать URL в HTML
     * 
     * @param string $html HTML код
     * @param string $domain Домен
     * @return string
     */
    private static function absolutizeUrl(string $html, string $domain): string
    {
        // Замена относительных URL на абсолютные
        $html = preg_replace(
            '/(href|src|action)=["\']([^"\']+)/i',
            function($matches) use ($domain) {
                $attr = $matches[1];
                $url = $matches[2];
                
                if (strpos($url, 'http') === 0 || strpos($url, '//') === 0) {
                    return $matches[0];
                }
                
                if (strpos($url, '/') === 0) {
                    return $attr . '="https://' . $domain . $url;
                }
                
                return $attr . '="https://' . $domain . '/' . $url;
            },
            $html
        );

        return $html;
    }
}

