<?php
/**
 * Класс для кеширования цен
 * Оптимизирован для PHP 8.1+
 * 
 * @package Sport38\Kant
 */

namespace Sport38\Kant;

class PriceCache
{
    private static array $priceDataCache = [];

    /**
     * Сброс кеша при изменении скидки
     * 
     * @param int|array $ID ID скидки или массив полей
     * @param array|null $arFields Поля скидки
     * @param mixed $offset Смещение
     * @return void
     */
    public static function discountReset($ID, ?array $arFields = null, $offset = null): void
    {
        if ($arFields === null && is_array($ID)) {
            $arFields = $ID;
            $ID = $arFields["ID"] ?? 0;
        }

        $now = time();

        // Пропуск, если есть купоны
        if (!empty($arFields["CATALOG_COUPONS"]) || !empty($arFields["COUPON"])) {
            return;
        }

        if ($ID > 0 && \Bitrix\Main\Loader::includeModule("catalog")) {
            $rsCoupons = \CCatalogDiscountCoupon::GetList(
                false,
                ["DISCOUNT_ID" => $ID],
                false,
                ["nTopCount" => 1]
            );

            if ($rsCoupons->Fetch()) {
                return;
            }
        }

        PriceAgent::start($arFields["SITE_ID"] ?? SITE_ID);

        // Обработка дат активности
        if (isset($arFields["ACTIVE_FROM"])) {
            $activeFrom = MakeTimeStamp($arFields["ACTIVE_FROM"]);
            if ($activeFrom > $now) {
                PriceAgent::addAgent("start", $arFields["SITE_ID"] ?? SITE_ID, $arFields["ACTIVE_FROM"]);
            }
        }

        if (isset($arFields["ACTIVE_TO"])) {
            $activeTo = MakeTimeStamp($arFields["ACTIVE_TO"]);
            if ($activeTo > $now) {
                PriceAgent::addAgent("stop", $arFields["SITE_ID"] ?? SITE_ID, $arFields["ACTIVE_TO"]);
            }
        }
    }

    /**
     * Сохранить цену перед удалением
     * 
     * @param int $id ID цены
     * @return void
     */
    public static function savePrice(int $id): void
    {
        if ($id <= 0 || !\Bitrix\Main\Loader::includeModule('catalog')) {
            return;
        }

        $query = \CPrice::GetList(
            [],
            ['ID' => $id],
            false,
            false,
            ['ID', 'PRODUCT_ID', 'CATALOG_GROUP_ID']
        );

        if ($price = $query->Fetch()) {
            self::$priceDataCache[$id] = $price;
        }
    }

    /**
     * Кешировать цены товара
     * 
     * @param int|array $ID ID товара или массив полей
     * @param array|false $arFields Поля товара
     * @param bool $skipActive Пропустить проверку активности
     * @return void
     */
    public static function cache($ID, $arFields = false, bool $skipActive = false): void
    {
        if (!$arFields && is_array($ID)) {
            $arFields = $ID;
            $ID = $arFields["ID"] ?? $arFields["RESULT"] ?? 0;
        } elseif (!$arFields && isset(self::$priceDataCache[$ID])) {
            $arFields = self::$priceDataCache[$ID];
        }

        if ($ID <= 0 || !\Bitrix\Main\Loader::includeModule("catalog")) {
            return;
        }

        // Пропуск при импорте из 1С
        if (Import::is1CSession()) {
            return;
        }

        $productId = (int)$ID;
        if ($productId <= 0) {
            return;
        }

        // Получение цен
        if (!\Bitrix\Main\Loader::includeModule("iblock")) {
            return;
        }

        $priceId = class_exists('\Sport38\Kant\Price') ? \Sport38\Kant\Price::getID() : (defined("PRICE_ID") ? (int)PRICE_ID : 1);
        $prices = \CIBlockPriceTools::GetCatalogPrices($productId, [$priceId]);
        
        if (empty($prices) || !is_array($prices)) {
            return;
        }

        // Сохранение в свойство PRICE_CACHE
        $priceData = serialize($prices);
        $iblockId = (int)($arFields["IBLOCK_ID"] ?? 0);
        
        if ($iblockId > 0) {
            \CIBlockElement::SetPropertyValues(
                $productId,
                $iblockId,
                $priceData,
                "PRICE_CACHE"
            );
        }
    }

    /**
     * Очистить кеш цен
     */
    public static function clearCache(): void
    {
        self::$priceDataCache = [];
    }
}

