<?php
/**
 * Класс для обработки изображений товаров
 * Оптимизирован для PHP 8.1+
 * 
 * @package Sport38\Kant
 */

namespace Sport38\Kant;

class ProductImg
{
    private static array $size = [
        "width" => 260,
        "height" => 260
    ];
    private static string $rule = "resample";
    private static int $compression = 80;

    /**
     * Обработать изображение товара
     * 
     * @param array $fields Поля элемента
     * @return array Обработанные поля
     */
    public static function handle(array &$fields): array
    {
        $catalogId = defined("CATALOG_IBLOCK_ID") ? (int)CATALOG_IBLOCK_ID : 0;
        
        if ($catalogId > 0 && isset($fields["IBLOCK_ID"]) && (int)$fields["IBLOCK_ID"] === $catalogId) {
            $img = self::extractPic($fields["DETAIL_PICTURE"] ?? null);

            if (is_array($img) && !empty($img)) {
                if (!isset($fields["PROPERTY_VALUES"])) {
                    $fields["PROPERTY_VALUES"] = [];
                }

                $resized = self::resize($img);
                if ($resized) {
                    $fields["PROPERTY_VALUES"]["MIDDLE_PICTURE"] = $resized;
                }
            }
        }

        return $fields;
    }

    /**
     * Изменить размер изображения
     * 
     * @param array $img Массив изображения
     * @return array|false Результат или false
     */
    public static function resize(array $img)
    {
        if (empty($img["tmp_name"]) || !file_exists($img["tmp_name"])) {
            return false;
        }

        $size = \CFile::GetImageSize($img["tmp_name"]);
        if (!$size) {
            return false;
        }

        list($width, $height) = $size;

        // Если изображение меньше нужного размера, возвращаем как есть
        if ($width <= self::$size["width"] && $height <= self::$size["height"]) {
            return $img;
        }

        $img["COPY_FILE"] = "Y";

        return \CIBlock::ResizePicture(
            $img,
            [
                "WIDTH" => self::$size["width"],
                "HEIGHT" => self::$size["height"],
                "METHOD" => self::$rule,
                "COMPRESSION" => self::$compression
            ]
        );
    }

    /**
     * Извлечь изображение из массива
     * 
     * @param mixed $pics Изображение(я)
     * @return array|null Массив изображения или null
     */
    private static function extractPic($pics): ?array
    {
        if (!is_array($pics)) {
            return null;
        }

        // Если это уже готовый массив с tmp_name
        if (isset($pics["tmp_name"])) {
            return $pics;
        }

        // Рекурсивный поиск
        foreach ($pics as $pic) {
            $result = self::extractPic($pic);
            if ($result !== null) {
                return $result;
            }
        }

        return null;
    }

    /**
     * Установить размеры для ресайза
     * 
     * @param int $width Ширина
     * @param int $height Высота
     * @return void
     */
    public static function setSize(int $width, int $height): void
    {
        self::$size = [
            "width" => max(1, $width),
            "height" => max(1, $height)
        ];
    }
}

