<?php
/**
 * Класс для работы с городами
 * Оптимизирован для PHP 8.1+ с улучшенным кешированием
 * 
 * @package Sport38\Kant
 */

namespace Sport38\Kant;

class City
{
    private static ?int $iblockId = null;
    private const CACHE_TIME = 3600;
    private const CACHE_DIR = "sport38.kant.city";

    /**
     * Найти город по ID
     * 
     * @param int $id ID города
     * @return array|false Данные города или false
     */
    public static function findById(int $id)
    {
        if ($id <= 0) {
            return false;
        }

        return self::find(
            ["ID" => $id],
            ["nTopCount" => 1]
        );
    }

    /**
     * Найти город по умолчанию
     * 
     * @return array|false Данные города или false
     */
    public static function findDefault()
    {
        return self::find(
            ["!DEFAULT" => false],
            ["nTopCount" => 1]
        );
    }

    /**
     * Найти города по фильтру
     * 
     * @param array $filter Фильтр
     * @param array|false $nav Навигация
     * @return array|false Данные города(ов) или false
     */
    public static function find(array $filter, $nav = false)
    {
        $filter = self::clearFilter($filter);
        $iblockId = self::getIBlockID();

        if ($iblockId <= 0) {
            return false;
        }

        $cache = new \CPHPCache();
        $cacheId = md5(serialize($filter) . serialize($nav) . $iblockId);

        if ($cache->InitCache(self::CACHE_TIME, $cacheId, self::CACHE_DIR)) {
            return $cache->GetVars();
        }

        if (!\Bitrix\Main\Loader::includeModule("iblock")) {
            return false;
        }

        // Получить список свойств для выборки
        $select = ["ID", "NAME", "CODE"];
        $dbProp = \CIBlockProperty::GetList([], [
            "IBLOCK_ID" => $iblockId
        ]);

        while ($prop = $dbProp->Fetch()) {
            $select[] = "PROPERTY_" . $prop["CODE"];
        }

        // Подготовка фильтра
        $filter = array_merge([
            "ACTIVE" => "Y",
            "ACTIVE_DATE" => "Y",
            "IBLOCK_ID" => $iblockId
        ], $filter);

        // Получение данных
        $elements = [];
        $dbElement = \CIBlockElement::GetList([], $filter, false, $nav, $select);

        while ($element = $dbElement->Fetch()) {
            // Нормализация свойств
            foreach ($element as $key => $value) {
                if (preg_match("/^PROPERTY_(.*)_VALUE$/i", $key, $match)) {
                    $element[$match[1]] = $value;
                    unset($element[$key]);
                }
            }

            $elements[(int)$element["ID"]] = $element;
        }

        // Сохранение в кеш
        $cache->StartDataCache();
        $cache->EndDataCache($elements);

        if (defined("BX_COMP_MANAGED_CACHE")) {
            global $CACHE_MANAGER;
            $CACHE_MANAGER->StartTagCache(self::CACHE_DIR);
            $CACHE_MANAGER->RegisterTag("iblock_id_" . $iblockId);
            $CACHE_MANAGER->EndTagCache();
        }

        // Возврат результата
        if (count($elements) === 1) {
            return reset($elements);
        }

        return !empty($elements) ? $elements : false;
    }

    /**
     * Получить список городов
     * 
     * @param array $filter Фильтр
     * @return array Массив городов
     */
    public static function getList(array $filter = []): array
    {
        $elements = self::find($filter);

        if (is_array($elements) && isset($elements["ID"])) {
            return [$elements["ID"] => $elements];
        }

        return is_array($elements) ? $elements : [];
    }

    /**
     * Очистить фильтр от null значений
     * 
     * @param array $filter Фильтр
     * @return array Очищенный фильтр
     */
    private static function clearFilter(array $filter): array
    {
        return array_filter($filter, function($value) {
            return $value !== null;
        });
    }

    /**
     * Получить ID инфоблока городов
     * 
     * @return int ID инфоблока
     */
    public static function getIBlockID(): int
    {
        if (self::$iblockId === null) {
            self::$iblockId = defined("CITY_IBLOCK_ID") ? (int)CITY_IBLOCK_ID : 0;
        }
        return self::$iblockId;
    }
}

