<?php
/**
 * Класс для работы с цветами
 * Оптимизирован для PHP 8.1+ с улучшенным кешированием
 * 
 * @package Sport38\Kant
 */

namespace Sport38\Kant;

class Color
{
    private const CACHE_TIME = 36000;
    private const CACHE_DIR = "sport38.kant.color";
    private const HLBLOCK_TABLE = "color_reference";

    /**
     * Получить цвета по XML_ID
     * 
     * @param array|string $ids XML_ID цветов
     * @param bool $needCache Нужно ли кешировать
     * @return array Массив цветов
     */
    public static function getByXmlID($ids, bool $needCache = true): array
    {
        // Нормализация входных данных
        if (!is_array($ids)) {
            $ids = [$ids];
        }
        
        $ids = array_filter($ids, function($id) {
            return !empty($id);
        });

        if (empty($ids)) {
            return [];
        }

        $result = [];
        $cache = new \CPHPCache();
        $cacheId = 'sport38_color_' . md5(implode("_", $ids));

        if ($cache->InitCache(self::CACHE_TIME, $cacheId, self::CACHE_DIR)) {
            return $cache->GetVars();
        }

        if (!\Bitrix\Main\Loader::includeModule('highloadblock')) {
            return [];
        }

        try {
            $hlblock = \Bitrix\Highloadblock\HighloadBlockTable::getList([
                'select' => ['TABLE_NAME', 'NAME', 'ID'],
                'filter' => ['=TABLE_NAME' => self::HLBLOCK_TABLE]
            ])->fetch();

            if (empty($hlblock['ID'])) {
                return [];
            }

            $entity = \Bitrix\Highloadblock\HighloadBlockTable::compileEntity($hlblock);
            $entityDataClass = $entity->getDataClass();

            $rsData = $entityDataClass::getList([
                "filter" => [
                    "LOGIC" => "OR",
                    ["UF_XML_ID" => $ids],
                    ["UF_XML_ID_EN" => $ids]
                ]
            ]);

            while ($arData = $rsData->fetch()) {
                $resultData = [
                    "NAME" => $arData["UF_NAME"] ?? "",
                    "COLOR" => $arData["UF_COLOR"] ?? "",
                    "HIGHLIGHT" => $arData["UF_HIGHLIGHT"] ?? "",
                    "UF_XML_ID" => $arData["UF_XML_ID"] ?? ""
                ];

                if (!empty($arData["UF_FILE"])) {
                    $fileId = (int)$arData["UF_FILE"];
                    if ($fileId > 0) {
                        $resultData["IMG"] = \CFile::GetFileArray($fileId);
                    }
                }

                // Индексация по обоим XML_ID
                $keys = ["UF_XML_ID", "UF_XML_ID_EN"];
                foreach ($keys as $key) {
                    if (!empty($arData[$key])) {
                        $result[$arData[$key]] = $resultData;
                    }
                }
            }
        } catch (\Exception $e) {
            // Логирование ошибки (если нужно)
            return [];
        }

        if ($needCache && !empty($result)) {
            $cache->StartDataCache();
            $cache->EndDataCache($result);
        }

        return $result;
    }

    /**
     * Получить цвет по XML_ID
     * 
     * @param string $xmlId XML_ID цвета
     * @return array|false Данные цвета или false
     */
    public static function getByXmlIDSingle(string $xmlId)
    {
        $result = self::getByXmlID($xmlId);
        return $result[$xmlId] ?? false;
    }
}

