<?php
/**
 * Класс для обработки атрибутов при импорте
 * Оптимизирован для PHP 8.1+
 * 
 * @package Sport38\Kant
 */

namespace Sport38\Kant;

class ImportAttr
{
    /**
     * Обработать атрибуты элемента
     * 
     * @param array $fields Поля элемента
     * @return void
     */
    public static function handle(array $fields): void
    {
        if (
            empty($fields["RESULT"]) ||
            !Import::is1CSession() ||
            !Import::isOffer((int)($fields["IBLOCK_ID"] ?? 0))
        ) {
            return;
        }

        self::update($fields);
    }

    /**
     * Обновить атрибуты
     * 
     * @param array $fields Поля элемента
     * @param bool $debug Режим отладки
     * @return void
     */
    public static function update(array $fields, bool $debug = false): void
    {
        $iblockID = (int)($fields["IBLOCK_ID"] ?? 0);
        $id = (int)($fields["ID"] ?? $fields["RESULT"] ?? 0);

        if ($iblockID <= 0 || $id <= 0) {
            return;
        }

        $attrs = \CIBlockElement::GetProperty(
            $iblockID,
            $id,
            "sort",
            "asc",
            ["CODE" => "CML2_ATTRIBUTES"]
        );

        while ($attr = $attrs->Fetch()) {
            $attrValue = $attr["VALUE"] ?? "";
            $attrName = $attr["DESCRIPTION"] ?? "";
            $attrCode = self::name2Code($attrName);

            if (empty($attrCode)) {
                continue;
            }

            // Поиск или создание свойства
            $attrPropID = self::findOrCreateProperty($attrName, $attrCode, $iblockID);

            if ($attrPropID > 0) {
                \CIBlockElement::SetPropertyValuesEx(
                    $id,
                    $iblockID,
                    [$attrCode => $attrValue]
                );
            }
        }
    }

    /**
     * Преобразовать имя в код
     * 
     * @param string $name Имя
     * @return string Код
     */
    private static function name2Code(string $name): string
    {
        $code = \CUtil::translit(
            $name,
            "ru",
            [
                "max_len" => 50,
                "change_case" => "U",
                "replace_space" => "_",
                "replace_other" => "_",
                "delete_repeat_replace" => true,
                "use_google" => false
            ]
        );

        return $code;
    }

    /**
     * Найти или создать свойство
     * 
     * @param string $name Имя
     * @param string $code Код
     * @param int $iblockID ID инфоблока
     * @return int ID свойства
     */
    private static function findOrCreateProperty(string $name, string $code, int $iblockID): int
    {
        $dbProp = \CIBlockProperty::GetList(
            [],
            [
                "IBLOCK_ID" => $iblockID,
                "XML_ID" => $code
            ]
        );

        if ($prop = $dbProp->Fetch()) {
            return (int)$prop["ID"];
        }

        // Создание свойства
        $obProp = new \CIBlockProperty;
        $propFields = [
            "IBLOCK_ID" => $iblockID,
            "NAME" => $name,
            "CODE" => $code,
            "XML_ID" => $code,
            "PROPERTY_TYPE" => "S",
            "ACTIVE" => "Y"
        ];

        $propId = $obProp->Add($propFields);
        return $propId ? (int)$propId : 0;
    }
}

