<?php
/**
 * Класс для обработки элементов при импорте
 * Оптимизирован для PHP 8.1+
 * 
 * @package Sport38\Kant
 */

namespace Sport38\Kant;

class ImportElement
{
    private static array $requiredFields = [
        "DETAIL_PICTURE",
        "DETAIL_TEXT",
        "PROPERTY_CAN_BUY"
    ];

    private static array $props = [
        "PROPERTY_HIDE_ON_SITE"
    ];

    /**
     * Обработать активность элемента
     * 
     * @param array $fields Поля элемента
     * @param bool $skipPrice Пропустить обновление цен
     * @return void
     */
    public static function handleActive(array $fields, bool $skipPrice = false): void
    {
        if (
            ($fields["sport_38_skip_active"] ?? '') === "Y" ||
            empty($fields["RESULT"]) ||
            !Import::isCatalog((int)($fields["IBLOCK_ID"] ?? 0)) ||
            !\Bitrix\Main\Loader::includeModule("iblock")
        ) {
            return;
        }

        $id = (int)($fields["ID"] ?? $fields["RESULT"] ?? 0);
        if ($id <= 0) {
            return;
        }

        if (!$skipPrice) {
            PriceCache::cache($id, $fields, true);
        }

        // Получение данных элемента
        $select = array_merge(
            ["IBLOCK_ID", "ID", "ACTIVE"],
            self::$requiredFields,
            self::$props
        );

        $db = \CIBlockElement::GetList(
            [],
            [
                "IBLOCK_ID" => (int)$fields["IBLOCK_ID"],
                "ID" => $id
            ],
            false,
            ["nTopCount" => 1],
            $select
        );

        if (!$item = $db->Fetch()) {
            return;
        }

        // Проверка обязательных полей
        $isValid = true;
        foreach (self::$requiredFields as $field) {
            $fieldValue = $item[$field] ?? $item[$field . "_VALUE"] ?? null;
            if (!self::checkValue($fieldValue)) {
                $isValid = false;
                break;
            }
        }

        // Проверка свойства HIDE_ON_SITE
        $hideOnSite = $item["PROPERTY_HIDE_ON_SITE_VALUE"] ?? null;
        if ($hideOnSite === "Y") {
            $isValid = false;
        }

        // Обновление активности
        $targetActive = $isValid ? "Y" : "N";
        if (($item["ACTIVE"] ?? "") !== $targetActive) {
            $ob = new \CIBlockElement;
            $ob->Update($id, ["ACTIVE" => $targetActive]);
        }
    }

    /**
     * Обработать обновление элемента
     * 
     * @param array $fields Поля элемента
     * @return void
     */
    public static function handleUpdate(array &$fields): void
    {
        // Логика обработки обновления
        // Можно добавить дополнительную валидацию
    }

    /**
     * Проверить значение поля
     * 
     * @param mixed $value Значение
     * @return bool
     */
    private static function checkValue($value): bool
    {
        if ($value === null || $value === false) {
            return false;
        }

        if (is_string($value)) {
            return trim($value) !== '';
        }

        if (is_array($value)) {
            return !empty($value);
        }

        return true;
    }
}

