<?php
/**
 * Класс для работы с ценами
 * Оптимизирован для PHP 8.1+ и нового шаблона
 * 
 * @package Sport38\Kant
 */

namespace Sport38\Kant;

class Price
{
    /**
     * Получить оптимальную цену из массива цен
     * 
     * @param array $prices Массив цен
     * @param bool $invert Инвертировать выбор (максимальная вместо минимальной)
     * @return array|false Массив с ценой или false
     */
    public static function getOptimal(array $prices = [], bool $invert = false)
    {
        if (!\Bitrix\Main\Loader::includeModule("currency")) {
            return false;
        }

        $baseCurrency = \CCurrency::GetBaseCurrency();
        if (empty($baseCurrency)) {
            return false;
        }

        if (empty($prices) || !is_array($prices)) {
            return false;
        }

        $minPrice = -1;
        $optimalPrice = [];

        foreach ($prices as $priceData) {
            if (!is_array($priceData) || empty($priceData)) {
                continue;
            }

            $resultPrice = [
                'PRICE' => 0,
                'DISCOUNT_PRICE' => 0,
                'CURRENCY' => $baseCurrency,
                'CAN_BUY' => $priceData['CAN_BUY'] ?? 'Y'
            ];

            // Конвертация в базовую валюту
            if (($priceData['CURRENCY'] ?? '') == $baseCurrency) {
                $resultPrice['PRICE'] = floatval($priceData['VALUE'] ?? 0);
                $resultPrice['DISCOUNT_PRICE'] = floatval($priceData['DISCOUNT_VALUE'] ?? $priceData['VALUE'] ?? 0);
            } else {
                $resultPrice['PRICE'] = \CCurrencyRates::ConvertCurrency(
                    floatval($priceData['VALUE'] ?? 0),
                    $priceData['CURRENCY'] ?? $baseCurrency,
                    $baseCurrency
                );
                $resultPrice['DISCOUNT_PRICE'] = \CCurrencyRates::ConvertCurrency(
                    floatval($priceData['DISCOUNT_VALUE'] ?? $priceData['VALUE'] ?? 0),
                    $priceData['CURRENCY'] ?? $baseCurrency,
                    $baseCurrency
                );
            }

            // Выбор оптимальной цены
            if (
                $minPrice == -1 ||
                (!$invert && $minPrice > $resultPrice['DISCOUNT_PRICE']) ||
                ($invert && $minPrice < $resultPrice['DISCOUNT_PRICE'])
            ) {
                $minPrice = $resultPrice['DISCOUNT_PRICE'];
                $optimalPrice = array_merge($priceData, $resultPrice);
            }
        }

        return !empty($optimalPrice) ? $optimalPrice : false;
    }

    /**
     * Получить ID типа цены
     * 
     * @return int ID типа цены
     */
    public static function getID(): int
    {
        return defined("PRICE_ID") ? (int)PRICE_ID : 1;
    }

    /**
     * Получить список типов цен
     * 
     * @param int|false $id ID типа цены (если false, используется getID())
     * @return array Массив типов цен
     */
    public static function getPrices($id = false): array
    {
        if ($id === false) {
            $id = self::getID();
        }

        $prices = [];

        if (!\Bitrix\Main\Loader::includeModule("catalog")) {
            return $prices;
        }

        $catalogGroups = \CCatalogGroup::GetListArray();
        foreach ($catalogGroups as $group) {
            if ((int)$group["ID"] == (int)$id) {
                $prices[$group["NAME"]] = [
                    "ID" => (int)$group["ID"],
                    "TITLE" => htmlspecialcharsbx($group["NAME_LANG"] ?? $group["NAME"]),
                    "SELECT" => "CATALOG_GROUP_" . $group["ID"],
                    "CAN_VIEW" => "Y",
                    "CAN_BUY" => "Y",
                ];
            }
        }

        return $prices;
    }

    /**
     * Форматировать цену для отображения
     * 
     * @param float $price Цена
     * @param string $currency Валюта
     * @return string Отформатированная цена
     */
    public static function format(float $price, string $currency = 'RUB'): string
    {
        if (!\Bitrix\Main\Loader::includeModule("currency")) {
            return number_format($price, 2, '.', ' ') . ' ' . $currency;
        }

        return \CCurrencyLang::CurrencyFormat($price, $currency, true);
    }

    /**
     * Рассчитать процент скидки
     * 
     * @param float $originalPrice Оригинальная цена
     * @param float $discountPrice Цена со скидкой
     * @return int Процент скидки (0-100)
     */
    public static function calculateDiscountPercent(float $originalPrice, float $discountPrice): int
    {
        if ($originalPrice <= 0 || $discountPrice >= $originalPrice) {
            return 0;
        }

        return (int)round((($originalPrice - $discountPrice) * 100) / $originalPrice);
    }
}

